/*
 * If distributed as part of the Linux kernel, this code is licensed under the
 * terms of the GPL v2.
 *
 * Otherwise, the following license terms apply:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1) Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3) The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGfsES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#include <linux/tty.h>
#include <linux/moduleparam.h>
#include <linux/device.h>
#include <linux/fs.h>			// iminor()
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/err.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include "fcucom.h"
#include "fcuvar.h"
#include "fax_share.h"		/* XXX */
#include "fcu_subr.h"

#define DRIVER_NAME	"fcud"		/* デバイス名 */

extern ulong A53_R4_BOUNDARY_ADDRESS;

/*
 * FBI2不具合対策
 */
#define FCUD_FBI2_CHECK_SIZE	3072


/*
 * sc_flags
 */
#define FSF_ISOPEN				0x0001		/* open されている */
#define FSF_RSLEEP				0x0002		/* DMA終了待ちで sleep 中 */
#define FSF_RSELWANT			0x0004		/* DMA終了待ち select 中 */

/*
 * DMAハンドラmacro
 */
#define	fcud_get_dma_handler(ch)					\
	(ch < FCUD_CH_MAX ? &sc->sc_dma[ch] : NULL)


/*
 * DMAチャンネル毎のDMA実行関数
 */
struct fcud_softc;
typedef struct fcud_dmah_tag {
	void			(*dma_start)(struct fcud_softc *, struct fcud_dmah_tag *);
	int				(*dma_abort)(struct fcud_softc *, struct fcud_dmah_tag *);
	void			(*dma_rsp)(struct fcud_softc *, struct fcud_dmah_tag *);
	int				(*dma_intr)(struct fcud_softc *, u_long);
	void			(*dma_abtrq)(struct fcud_softc *);
        dma_addr_t      bus_addr;
        size_t          size;
} fcud_dmah_t;

struct fcud_softc {
		fax_driver_header_t		sc_header;
        struct pci_dev                  *pci_dev;
        u_short                         sc_flags;       /* flags */
        u_short                         sc_afail;       /* attach failed */
        u_short                         sc_run;         /* DMA実行中チャンネル */
        u_short                         sc_rsp;         /* DMA実行終了レスポンス */
        fcud_dmah_t                     sc_dma[FCUD_CH_MAX]; /* DMAハンドラ */
        fcud_cmd_t                      sc_wcmd;        /* command work buffer */
        fcud_cmd_t                      sc_rrsp;        /* response work buffer */
        wait_queue_head_t               sc_rpl_q;
        unsigned int                    irqno;
        unsigned long                   io_vaddr;
        unsigned long                   io_addr;
        unsigned long                   io_len;
        unsigned long                   mem_vaddr;
        unsigned long                   mem_addr;
        unsigned long                   mem_len;
        fax_share_param_t               *sc_share;
};
typedef struct fcud_softc	fcud_t;

fcud_t g_fcud_softc;
EXPORT_SYMBOL(g_fcud_softc);
static fcud_t *sc = &g_fcud_softc;

static int fcudopen( struct inode *inode, struct file *filp );
static int fcudclose( struct inode *inode, struct file *filp );
static long fcudioctl( struct file *filp, unsigned int cmd, unsigned long arg);
static ssize_t fcudread( struct file* p_file, char* buf, size_t count, loff_t* f_pos );
static ssize_t fcudwrite( struct file* p_file, const char* buf, size_t count, loff_t* f_pos );

struct file_operations fcud_FileOps = {
 .owner				= THIS_MODULE,
 .unlocked_ioctl		= fcudioctl,
 .open				= fcudopen,
 .read				= fcudread,
 .write				= fcudwrite,
 .release			= fcudclose,
};
EXPORT_SYMBOL(fcud_FileOps);


void	fcud_dma_start_scan(fcud_t *, fcud_dmah_t *);
void	fcud_dma_start_plot(fcud_t *, fcud_dmah_t *);
void	fcud_dma_start_send(fcud_t *, fcud_dmah_t *);
void	fcud_dma_start_recv(fcud_t *, fcud_dmah_t *);
int		fcud_dma_abort_scan(fcud_t *, fcud_dmah_t *);
int		fcud_dma_abort_plot(fcud_t *, fcud_dmah_t *);
int		fcud_dma_abort_send(fcud_t *, fcud_dmah_t *);
int		fcud_dma_abort_recv(fcud_t *, fcud_dmah_t *);
void	fcud_dma_rsp_scan(fcud_t *, fcud_dmah_t *);
void	fcud_dma_rsp_plot(fcud_t *, fcud_dmah_t *);
void	fcud_dma_rsp_send(fcud_t *, fcud_dmah_t *);
void	fcud_dma_rsp_recv(fcud_t *, fcud_dmah_t *);
int		fcud_dma_intr_scan(fcud_t *, u_long);
int		fcud_dma_intr_plot(fcud_t *, u_long);
int		fcud_dma_intr_send(fcud_t *, u_long);
int		fcud_dma_intr_recv(fcud_t *, u_long);
void	fcud_dma_abtrq_send(fcud_t *);
void	fcud_dma_abtrq_recv(fcud_t *);

int	fcud_attach		(void *);
int	fcud_detach		(void *);
irqreturn_t	fcudintr	(void *);
static void fcud_initial(void);

#ifdef	FCUD_DEBUG_IOCTL
void	fcud_debug_0(fcud_t *);
void	fcud_debug_ioctl_func(fcud_t *, int);
#endif	/* !FCUD_DEBUG_IOCTL */
/*
 * DMA start
 */
void
fcud_dma_start_scan(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long srcAddr, srcSize, srcWidth, dstAddr, dstSize, dstWidth, line;
	u_short whiteTop, whiteBottom;
	u_short i;
	u_char *srcVaddr, *dstVaddr, *srcVaddrOrg, *dstVaddrOrg;

	/* 共通 */
	line = sc->sc_wcmd.line;
	PRINT_INFO("DMA SCAN line=%lx\n",line);
	
	/* 転送元 */
	/* 転送元サイズ */
	/* widthはdot単位で通知されるため、byte単位に変換 */
	srcWidth = sc->sc_wcmd.width / 8;
	srcSize = srcWidth * line;
	/* 転送元アドレス */
	srcAddr = sc->sc_wcmd.offset;
	PRINT_INFO("DMA SCAN srcAdr=%p srcWidth=%lx, srcSize=%lx\n", (void *)srcAddr, srcWidth, srcSize);
	/* アドレスの妥当性チェック */
	if((srcAddr+srcSize-1) >= A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("scan src address is out of range, srcAddr=%p srcSize=%lx\n", (void *)srcAddr, srcSize);
	}

	/* 転送元メモリマッピング */
#ifdef FCUD_CACHE_USE
	srcVaddrOrg = (u_char *)ioremap_cache(srcAddr, srcSize);
#else
	srcVaddrOrg = (u_char *)ioremap_nocache(srcAddr, srcSize);
#endif
	if(!srcVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		return;
	}
	PRINT_INFO("DMA SCAN srcVaddrOrg=%p\n", (void *)srcVaddrOrg);
	srcVaddr = srcVaddrOrg;
	
	/* 転送先 */
	/* 先端白画素・後端白画素情報 */
	/* キャッシュインバリデートの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
	whiteTop = cpu_to_be16(bsr2_fcubase(sc, FBI_OFS_DP_SCAN_WHITE_TOP));
	whiteBottom = cpu_to_be16(bsr2_fcubase(sc, FBI_OFS_DP_SCAN_WHITE_BOTTOM));
	PRINT_INFO("DMA SCAN whiteTop=%d\n", whiteTop);
	PRINT_INFO("DMA SCAN whiteBottom=%d\n", whiteBottom);
	/* 転送先サイズ */
	dstWidth = srcWidth + whiteTop + whiteBottom;
	dstSize = dstWidth * line;
	/* 転送先アドレス */
	dstAddr = cpu_to_be32(bsr4_fcubase(sc, FBI_OFS_DP_SCAN_ADDR));
	PRINT_INFO("DMA SCAN dstAddr=%p dstWidth=%lx, dstSize=%lx\n", (void *)dstAddr, dstWidth, dstSize);

	/* アドレスの妥当性チェック */
	if(dstAddr < A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("scan dst address is out of range, address = %p\n", (void *)dstAddr);
	}

	/* メモリマッピング */
#ifdef FCUD_CACHE_USE
	dstVaddrOrg = (u_char *)ioremap_cache(dstAddr, dstSize);
#else
	dstVaddrOrg = (u_char *)ioremap_nocache(dstAddr, dstSize);
#endif
	if(!dstVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		iounmap((void *)srcVaddrOrg);
		return;
	}
	PRINT_INFO("DMA SCAN dstVaddrOrg=%p\n", (void *)dstVaddrOrg);
	dstVaddr = dstVaddrOrg;
	
	for(i=0; i<line; i++){
		// 先端に白画素を付加
		memset(dstVaddr, 0, whiteTop);
		dstVaddr = dstVaddr + whiteTop;
		// メモリコピーする
		memcpy(dstVaddr, srcVaddr, srcWidth);
		dstVaddr = dstVaddr + srcWidth;
		srcVaddr = srcVaddr + srcWidth;
		// 後端に白画素を付加
		memset(dstVaddr, 0, whiteBottom);
		dstVaddr = dstVaddr + whiteBottom;
	}

#ifdef FCUD_CACHE_USE
	/* 転送先データエリアのキャッシュフラッシュを実施 */
	cache_flush_range(dstVaddrOrg, dstSize);
#endif

	/* メモリアンマッピング */
	iounmap((void *)srcVaddrOrg);
	iounmap((void *)dstVaddrOrg);
	
}
EXPORT_SYMBOL(fcud_dma_start_scan);

void
fcud_dma_start_plot(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long srcAddr, srcSize, srcWidth, dstAddr, dstSize, dstWidth, line;
	u_short dustTop, dustBottom;
	u_short i;
	u_char *srcVaddr, *dstVaddr, *srcVaddrOrg, *dstVaddrOrg;

	/* 共通 */
	line = sc->sc_wcmd.line;
	PRINT_INFO("DMA PLOT line=%lx\n",line);
	
	/* 転送先 */
	/* 転送先サイズ */
	/* widthはdot単位で通知されるため、byte単位に変換 */
	dstWidth = sc->sc_wcmd.width / 8;
	dstSize = dstWidth * line;
	/* 転送先アドレス */
	dstAddr = sc->sc_wcmd.offset;
	PRINT_INFO("DMA PLOT dstAdr=%p dstWidth=%lx, dstSize=%lx\n", (void *)dstAddr, dstWidth, dstSize);

	/* アドレスの妥当性チェック */
	if((dstAddr+dstSize-1) >= A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("plot dst address is out of range, dstAddr=%p dstSize=%lx\n", (void *)dstAddr, dstSize);
	}

	/* 転送先メモリマッピング */
#ifdef FCUD_CACHE_USE
	dstVaddrOrg = (u_char*)ioremap_cache(dstAddr, dstSize);
#else
	dstVaddrOrg = (u_char*)ioremap_nocache(dstAddr, dstSize);
#endif
	if(!dstVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		return;
	}
	PRINT_INFO("DMA PLOT dstVaddrOrg=%p\n", (void *)dstVaddrOrg);
	dstVaddr = dstVaddrOrg;

	/* 転送元 */
	/* 先端捨て画素・後端捨て画素情報 */
	/* キャッシュインバリデートの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
	dustTop = cpu_to_be16(bsr2_fcubase(sc, FBI_OFS_DP_PLOT_DUST_TOP));
	dustBottom = cpu_to_be16(bsr2_fcubase(sc, FBI_OFS_DP_PLOT_DUST_BOTTOM));
	PRINT_INFO("DMA PLOT dustTop=%d\n", dustTop);
	PRINT_INFO("DMA PLOT dustBottom=%d\n", dustBottom);
	/* 転送元サイズ */
	srcWidth = dstWidth + dustTop + dustBottom;
	srcSize = srcWidth * line;
	/* 転送元アドレス */
	srcAddr = cpu_to_be32(bsr4_fcubase(sc, FBI_OFS_DP_PLOT_ADDR));
	PRINT_INFO("DMA PLOT srcAddr=%p srcWidth=%lx, srcSize=%lx\n", (void *)srcAddr, srcWidth, srcSize);

	/* アドレスの妥当性チェック */
	if(srcAddr < A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("plot src address is out of range, srcAddr=%p\n", (void *)srcAddr);
	}

	/* メモリマッピング */
#ifdef FCUD_CACHE_USE
	srcVaddrOrg = (u_char*)ioremap_cache(srcAddr, srcSize);
#else
	srcVaddrOrg = (u_char*)ioremap_nocache(srcAddr, srcSize);
#endif
	if(!srcVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		iounmap((void *)dstVaddrOrg);
		return;
	}
	PRINT_INFO("DMA PLOT srcVaddrOrg=%p\n", (void *)srcVaddrOrg);
	srcVaddr = srcVaddrOrg;
	
#ifdef FCUD_CACHE_USE
	/* 転送元データエリアのキャッシュインバリデートを実施 */
	cache_invalidate_range(srcVaddrOrg, srcSize);
#endif

	for(i=0; i<line; i++){
		// 先端は読み捨て
		srcVaddr = srcVaddr + dustTop;
		// メモリコピーする
		memcpy(dstVaddr, srcVaddr, dstWidth);
		dstVaddr = dstVaddr + dstWidth;
		srcVaddr = srcVaddr + dstWidth;
		// 後端は読み捨て
		srcVaddr = srcVaddr + dustBottom;
	}
	
	/* メモリアンマッピング */
	iounmap((void *)srcVaddrOrg);
	iounmap((void *)dstVaddrOrg);
	
}
EXPORT_SYMBOL(fcud_dma_start_plot);


void
fcud_dma_start_send(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long srcAddr, dstAddr, size;
	u_char *srcVaddrOrg, *dstVaddrOrg;

	/* 転送元 */
	/* 転送サイズ */
	size = sc->sc_wcmd.width;
	/* 転送元アドレス */
	srcAddr = sc->sc_wcmd.offset;
	PRINT_INFO("DMA SEND srcAdr=%08lx size=%ld\n", srcAddr, size);

	/* アドレスの妥当性チェック */
	if((srcAddr+size-1) >= A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("send src address is out of range, srcAddr=%08lx size=%ld\n", srcAddr, size);
	}

	/* 転送元メモリマッピング */
#ifdef FCUD_CACHE_USE
	srcVaddrOrg = (u_char*)ioremap_cache(srcAddr, size);
#else
	srcVaddrOrg = (u_char*)ioremap_nocache(srcAddr, size);
#endif
	if(!srcVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		return;
	}
	PRINT_INFO("DMA SEND srcVaddrOrg=%08lx\n", (unsigned long)srcVaddrOrg);

	/* 転送先 */
	/* 転送先アドレス */
	/* キャッシュインバリデートの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
	dstAddr = cpu_to_be32(bsr4_fcubase(sc, FBI_OFS_DP_SCU_TO_FCU_ADDR));
	PRINT_INFO("DMA SEND dstAddr=%08lx \n", dstAddr);

	/* アドレスの妥当性チェック */
	if(dstAddr < A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("send dst address is out of range, address = %08lx\n", dstAddr);
	}

	/* 転送先メモリマッピング */
#ifdef FCUD_CACHE_USE
	dstVaddrOrg = (u_char*)ioremap_cache(dstAddr, size);
#else
	dstVaddrOrg = (u_char*)ioremap_nocache(dstAddr, size);
#endif
	if(!dstVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		iounmap((void *)srcVaddrOrg);
		return;
	}
	PRINT_INFO("DMA SEND dstVaddrOrg=%08lx \n", (unsigned long)dstVaddrOrg);
	
	/* メモリコピーする */
	memcpy(dstVaddrOrg, srcVaddrOrg, size);
	
#ifdef FCUD_CACHE_USE
	/* 転送先データエリアのキャッシュフラッシュを実施 */
	cache_flush_range(dstVaddrOrg, size);
#endif
	
	/* メモリアンマッピング */
	iounmap((void *)srcVaddrOrg);
	iounmap((void *)dstVaddrOrg);
	
}
EXPORT_SYMBOL(fcud_dma_start_send);

void
fcud_dma_start_recv(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long srcAddr, dstAddr, size;
	u_char *srcVaddrOrg, *dstVaddrOrg;

	/* 転送先 */
	/* 転送サイズ */
	size = sc->sc_wcmd.width;
	/* 転送先アドレス */
	dstAddr = sc->sc_wcmd.offset;
	PRINT_INFO("DMA RECV dstAddr=%08lx size=%ld\n", dstAddr, size);

	/* アドレスの妥当性チェック */
	if((dstAddr+size-1) >= A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("rcv dst address is out of range, dstAddr=%08lx size=%ld\n", dstAddr, size);
	}

	/* 転送先メモリマッピング */
#ifdef FCUD_CACHE_USE
	dstVaddrOrg = (u_char*)ioremap_cache(dstAddr, size);
#else
	dstVaddrOrg = (u_char*)ioremap_nocache(dstAddr, size);
#endif
	if(!dstVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		return;
	}
	PRINT_INFO("DMA RECV dstVaddrOrg=%08lx\n", (unsigned long)dstVaddrOrg);

	/* 転送元 */
	/* 転送元アドレス */
	/* キャッシュインバリデートの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
	srcAddr = cpu_to_be32(bsr4_fcubase(sc, FBI_OFS_DP_FCU_TO_SCU_ADDR));
	PRINT_INFO("DMA RECV srcAddr=%08lx \n", srcAddr);

	/* アドレスの妥当性チェック */
	if(srcAddr < A53_R4_BOUNDARY_ADDRESS){
		PRINT_EMERG("recv src address is out of range, srcAddr=%08lx\n", srcAddr);
	}

	/* 転送元メモリマッピング */
#ifdef FCUD_CACHE_USE
	srcVaddrOrg = (u_char*)ioremap_cache(srcAddr, size);
#else
	srcVaddrOrg = (u_char*)ioremap_nocache(srcAddr, size);
#endif
	if(!srcVaddrOrg){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		iounmap((void *)dstVaddrOrg);
		return;
	}
	PRINT_INFO("DMA RECV srcVaddrOrg=%08lx \n", (unsigned long)srcVaddrOrg);
	
#ifdef FCUD_CACHE_USE
	/* 転送元データエリアのキャッシュインバリデートを実施 */
	cache_invalidate_range(srcVaddrOrg, size);
#endif

	/* メモリコピーする */
#if 1
	memcpy(dstVaddrOrg, srcVaddrOrg, size);
#else
	{
		u_long cpysize;
		u_char top = 0;	
		u_char bottom = 0;
		u_char *src;
		u_char *dst;
	
		bottom = size % 4;
		if(bottom != 0){
			top = 4 - bottom;
		}
		cpysize = size - top - bottom;
		src = srcVaddrOrg;
		dst = dstVaddrOrg;
		
		PRINT_INFO("top=%d bottom=%d cpysize=%08lx", top, bottom, cpysize);
		
		if(top != 0){
			int i;
			for(i=0; i<top; i++){
				*dst++ = *src++;
			}
		}
		memcpy(dst, src, cpysize);
		if(bottom != 0){
			int i;
			for(i=0; i<bottom; i++){
				*dst++ = *src++;
			}
		}
	}	
#endif
	
	/* メモリアンマッピング */
	iounmap((void *)srcVaddrOrg);
	iounmap((void *)dstVaddrOrg);
	
}

EXPORT_SYMBOL(fcud_dma_start_recv);


/*
 * DMA abort
 * 止められなかったら 1
 * まだ割込みが起きていなかったら 0
 */
int
fcud_dma_abort_scan(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long intr;

	bsw4_io(sc, FBI_OFS_IO_DMACTL, FBI_BIT_DMACTL_DMA1);
	intr = bsr4_io(sc, FBI_OFS_IO_INTRCAUSE);
	/* 割込みが発生していなければsyncする */
	if (!FBI_IS_DMA1(intr)) {
//		dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_SCAN].bus_addr, sc->sc_dma[FCUD_CH_SCAN].size, DMA_TO_DEVICE);
		return (0);
	}
	return (1);
}
EXPORT_SYMBOL(fcud_dma_abort_scan);

int
fcud_dma_abort_plot(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long intr;

	bsw4_io(sc, FBI_OFS_IO_DMACTL, FBI_BIT_DMACTL_DMA2);
	intr = bsr4_io(sc, FBI_OFS_IO_INTRCAUSE);
	/* 割込みが発生していなければsyncする */
	if (!FBI_IS_DMA2(intr)) {
//		dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_PLOT].bus_addr, sc->sc_dma[FCUD_CH_PLOT].size, DMA_FROM_DEVICE);
		return (0);
	}
	return (1);
}
EXPORT_SYMBOL(fcud_dma_abort_plot);


int
fcud_dma_abort_send(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long intr;

	bsw4_io(sc, FBI_OFS_IO_DMACTL, FBI_BIT_DMACTL_DMA3A);
	intr = bsr4_io(sc, FBI_OFS_IO_INTRCAUSE);
	/* 割込みが発生していなければsyncする */
	if (!FBI_IS_DMA3A(intr)) {
//		dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_SEND].bus_addr, sc->sc_dma[FCUD_CH_SEND].size, DMA_TO_DEVICE);
		return (0);
	}
	return (1);
}
EXPORT_SYMBOL(fcud_dma_abort_send);

int
fcud_dma_abort_recv(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long intr;

	bsw4_io(sc, FBI_OFS_IO_DMACTL, FBI_BIT_DMACTL_DMA4A);
	intr = bsr4_io(sc, FBI_OFS_IO_INTRCAUSE);
	/* 割込みが発生していなければsyncする */
	if (!FBI_IS_DMA4A(intr)) {
//		dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_RECV].bus_addr, sc->sc_dma[FCUD_CH_RECV].size, DMA_FROM_DEVICE);
		return (0);
	}
	return (1);
}
EXPORT_SYMBOL(fcud_dma_abort_recv);


/*
 * DMA response
 */
void
fcud_dma_rsp_scan(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long x;

	x = bsr4_io(sc, FBI_OFS_IO_DMA1SIZE);
	sc->sc_rrsp.width = FBI_DMA12_WIDTH(x) * sizeof(long);
	sc->sc_rrsp.line  = FBI_DMA12_LINE(x);	/* -1して0にするか */
//	dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_SCAN].bus_addr, sc->sc_dma[FCUD_CH_SCAN].size, DMA_TO_DEVICE);
}
EXPORT_SYMBOL(fcud_dma_rsp_scan);


void
fcud_dma_rsp_plot(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long x;

	x = bsr4_io(sc, FBI_OFS_IO_DMA2SIZE);
	sc->sc_rrsp.width = FBI_DMA12_WIDTH(x) * sizeof(long);
	sc->sc_rrsp.line  = FBI_DMA12_LINE(x);	/* -1して0にするか */
//	dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_PLOT].bus_addr, sc->sc_dma[FCUD_CH_PLOT].size, DMA_FROM_DEVICE);
//	flush_cache_all();
}
EXPORT_SYMBOL(fcud_dma_rsp_plot);

void
fcud_dma_rsp_send(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long x;

	x = bsr4_io(sc, FBI_OFS_IO_DMA3SIZE);
	sc->sc_rrsp.width = FBI_DMA34_SIZE(x) * sizeof(long);
	sc->sc_rrsp.line  = 0;
//	dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_SEND].bus_addr, sc->sc_dma[FCUD_CH_SEND].size, DMA_TO_DEVICE);
}
EXPORT_SYMBOL(fcud_dma_rsp_send);

void
fcud_dma_rsp_recv(fcud_t *sc, fcud_dmah_t *dh)
{
	u_long x;

	x = bsr4_io(sc, FBI_OFS_IO_DMA4SIZE);
	sc->sc_rrsp.width = FBI_DMA34_SIZE(x) * sizeof(long);
	sc->sc_rrsp.line  = 0;
//	dma_unmap_single(&sc->pci_dev->dev, sc->sc_dma[FCUD_CH_RECV].bus_addr, sc->sc_dma[FCUD_CH_RECV].size, DMA_FROM_DEVICE);
//	flush_cache_all();
}
EXPORT_SYMBOL(fcud_dma_rsp_recv);

/*
 * DMA interrupt
 * 割込みが起きていたら1を返す
 */
int
fcud_dma_intr_scan(fcud_t *sc, u_long intr)
{
	if (FBI_IS_DMA1(intr)) {
		u_long mode;

		lpux_bsw4_io_intr_clear(sc,
								FBI_OFS_IO_DMAINTRCLR,
								FBI_BIT_DMAINTRCLR_D1);
		mode = bsr4_io(sc, FBI_OFS_IO_DMAMODE);
		mode &= ~FBI_BIT_DMAMODE_D1_ENA;
		bsw4_io(sc, FBI_OFS_IO_DMAMODE, mode);
		return (1);
	}
	return (0);
}
EXPORT_SYMBOL(fcud_dma_intr_scan);


int
fcud_dma_intr_plot(fcud_t *sc, u_long intr)
{
	if (FBI_IS_DMA2(intr)) {
		u_long mode;

		lpux_bsw4_io_intr_clear(sc,
								FBI_OFS_IO_DMAINTRCLR,
								FBI_BIT_DMAINTRCLR_D2);
		mode = bsr4_io(sc, FBI_OFS_IO_DMAMODE);
		mode &= ~FBI_BIT_DMAMODE_D2_ENA;
		bsw4_io(sc, FBI_OFS_IO_DMAMODE, mode);
		return (1);
	}
	return (0);
}
EXPORT_SYMBOL(fcud_dma_intr_plot);


int
fcud_dma_intr_send(fcud_t *sc, u_long intr)
{
	if (FBI_IS_DMA3A(intr)) {
		u_long mode;

		lpux_bsw4_io_intr_clear(sc,
								FBI_OFS_IO_DMAINTRCLR,
								FBI_BIT_DMAINTRCLR_D3A);
		mode = bsr4_io(sc, FBI_OFS_IO_DMAMODE);
		mode &= ~FBI_BIT_DMAMODE_D3AENA;
		bsw4_io(sc, FBI_OFS_IO_DMAMODE, mode);
		return (1);
	}
	return (0);
}
EXPORT_SYMBOL(fcud_dma_intr_send);

int
fcud_dma_intr_recv(fcud_t *sc, u_long intr)
{
	if (FBI_IS_DMA4A(intr)) {
		u_long mode;

		lpux_bsw4_io_intr_clear(sc,
								FBI_OFS_IO_DMAINTRCLR,
								FBI_BIT_DMAINTRCLR_D4A);
		mode = bsr4_io(sc, FBI_OFS_IO_DMAMODE);
		mode &= ~FBI_BIT_DMAMODE_D4AENA;
		bsw4_io(sc, FBI_OFS_IO_DMAMODE, mode);
		return (1);
	}
	return (0);
}
EXPORT_SYMBOL(fcud_dma_intr_recv);

void
fcud_dma_abtrq_send(fcud_t *sc)
{
#if 0 /* 動作しない */
	/* DMA stop confirm intr */
	bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_DMA3);
	bsw4_io(sc, FBI_OFS_IO_INTRCTL, FBI_BIT_INTRCTL_INTR);
#endif
}

void
fcud_dma_abtrq_recv(fcud_t *sc)
{
#if 0 /* 動作しない */
	/* DMA stop confirm intr */
	bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_DMA4);
	bsw4_io(sc, FBI_OFS_IO_INTRCTL, FBI_BIT_INTRCTL_INTR);
#endif
}

/********************************************************************/
static __inline void
fcud_dma_start(fcud_t *sc)
{
	fcud_dmah_t *dh;

	PRINT_INFO("fcud: fcud_dma_start cmd=0x%x, dmach=%d\n", sc->sc_wcmd.command, sc->sc_wcmd.dmach);

	if ((dh = fcud_get_dma_handler(sc->sc_wcmd.dmach)) != NULL) {
		if (dh->dma_start)
			(*dh->dma_start)(sc, dh);
// 		bit_set(sc->sc_run, sc->sc_wcmd.dmach);
	}
}


static __inline int
fcud_dma_abort(fcud_t *sc)
{
	fcud_dmah_t *dh;
	int toolate = 0;

	if ((dh = fcud_get_dma_handler(sc->sc_wcmd.dmach)) != NULL) {
		if (dh->dma_abort)
			if ((toolate = (*dh->dma_abort)(sc, dh)) == 0)
				bit_clr(sc->sc_run, sc->sc_wcmd.dmach);	/* 止めた */

	}
	return (toolate);
}


static __inline void
fcud_dma_rsp(fcud_t *sc)
{
	fcud_dmah_t *dh;
	u_short dmach;

	if (bit_is_set(sc->sc_rsp, FCUD_CH_RECV))	/* (1) */
		dmach = FCUD_CH_RECV;
	else if (bit_is_set(sc->sc_rsp, FCUD_CH_SEND))	/* (2) */
		dmach = FCUD_CH_SEND;
	else if (bit_is_set(sc->sc_rsp, FCUD_CH_SCAN))	/* (3) */
		dmach = FCUD_CH_SCAN;
	else if (bit_is_set(sc->sc_rsp, FCUD_CH_PLOT))	/* (4) */
		dmach = FCUD_CH_PLOT;
	else
		return;

	if ((dh = fcud_get_dma_handler(dmach)) != NULL) {
		sc->sc_rrsp.command = FCUD_RSP;
		sc->sc_rrsp.dmach   = dmach;
		if (dh->dma_rsp)
			(*dh->dma_rsp)(sc, dh);
		bit_clr(sc->sc_rsp, dmach);
	}
}


static __inline void
fcud_dma_intr(fcud_t *sc, fcud_dmah_t *dh, u_long intr, u_short dmach)
{
	if ((dh = fcud_get_dma_handler(dmach)) != NULL) {
		if (dh->dma_intr)
			if ((*dh->dma_intr)(sc, intr)) {
				bit_clr(sc->sc_run, dmach);
				bit_set(sc->sc_rsp, dmach);
				PRINT_INFO("[kernel thread] DMA xfer was finished.\n");
				if (FLAG_CHK(FSF_RSLEEP)) {
					FLAG_CLR(FSF_RSLEEP);
					wake_up_interruptible(&sc->sc_rpl_q);
				}
			}
	}
}


static __inline void
fcud_dma_abort_intr(fcud_t *sc, fcud_dmah_t *dh, u_short dmach)
{
	if ((dh = fcud_get_dma_handler(dmach)) != NULL) {
		if (dh->dma_abort)
			(*dh->dma_abort)(sc, dh);
		bit_clr(sc->sc_run, dmach);
		bit_set(sc->sc_rsp, dmach);
		if (dh->dma_abtrq)
			(*dh->dma_abtrq)(sc);
		if (FLAG_CHK(FSF_RSLEEP)) {
			FLAG_CLR(FSF_RSLEEP);
			wake_up_interruptible(&sc->sc_rpl_q);
		}
	}
}


static __inline void
fcud_dmamap_create(fcud_t *sc)
{
	fcud_dmah_t *dh;

	/*
	 * DMAハンドラとbus dma map
	 */

	/* FCUD_CH_SCAN */
	dh = fcud_get_dma_handler(FCUD_CH_SCAN);
	dh->dma_start = fcud_dma_start_scan;
	dh->dma_abort = fcud_dma_abort_scan;
	dh->dma_rsp   = fcud_dma_rsp_scan;
	dh->dma_intr  = fcud_dma_intr_scan;
	dh->dma_abtrq = NULL;

	/* FCUD_CH_PLOT */
	dh = fcud_get_dma_handler(FCUD_CH_PLOT);
	dh->dma_start = fcud_dma_start_plot;
	dh->dma_abort = fcud_dma_abort_plot;
	dh->dma_rsp   = fcud_dma_rsp_plot;
	dh->dma_intr  = fcud_dma_intr_plot;
	dh->dma_abtrq = NULL;

	/* FCUD_CH_SEND */
	dh = fcud_get_dma_handler(FCUD_CH_SEND);
	dh->dma_start = fcud_dma_start_send;
	dh->dma_abort = fcud_dma_abort_send;
	dh->dma_rsp   = fcud_dma_rsp_send;
	dh->dma_intr  = fcud_dma_intr_send;
	dh->dma_abtrq = fcud_dma_abtrq_send;

	/* FCUD_CH_RECV */
	dh = fcud_get_dma_handler(FCUD_CH_RECV);
	dh->dma_start = fcud_dma_start_recv;
	dh->dma_abort = fcud_dma_abort_recv;
	dh->dma_rsp   = fcud_dma_rsp_recv;
	dh->dma_intr  = fcud_dma_intr_recv;
	dh->dma_abtrq = fcud_dma_abtrq_recv;
}

/*
 * attach
 */
int fcud_attach(void *self)
{

	PRINT_INFO("fcud: fcud_attach was invoked.\n");

	/*
	 * init parameter
	 */
	fcud_initial();
	
	/* 
	 * get share param between fcu. 
	 */
	sc->sc_share = get_fax_share_param();
//	sc->pci_dev = sc->sc_share->pci_dev;
//	sc->io_vaddr = sc->sc_share->io_vaddr;
//	sc->io_addr = sc->sc_share->io_addr;
//	sc->io_len = sc->sc_share->io_len;
	sc->mem_vaddr = sc->sc_share->mem_vaddr;
	sc->mem_addr = sc->sc_share->mem_addr;
	sc->mem_len = sc->sc_share->mem_len;
//	sc->irqno = sc->sc_share->irqno;

	/* set dmamap */
	fcud_dmamap_create(sc);	
	
	/*
	 * regist interrupt handler … 削除
	 */

	PRINT_INFO("fcud attached\n");

	return 0;
}
EXPORT_SYMBOL(fcud_attach);

int fcud_detach(void *self)
{
	PRINT_INFO("fcud: fcud_detach was invoked.\n");
	memset(sc, 0, sizeof(fcud_t));
	return 0;
}
EXPORT_SYMBOL(fcud_detach);


/*
 * open
 */
int
fcudopen(struct inode* inode, struct file* p_file)
{
	PRINT_INFO("%s: device open\n", DRIVER_NAME);
	return (0);
}
EXPORT_SYMBOL(fcudopen);


/*
 * close
 */
int
fcudclose(struct inode* inode, struct file* p_file)
{

	PRINT_INFO("%s: device close\n", DRIVER_NAME);

	/*
	 * DMA割込みマスクとバイトレーンをリセット … 削除
	 */

	/*
	 * DMAを停止 … 削除
	 */

	/*
	 * 
	 */
	sc->sc_run = sc->sc_rsp = 0;

	/*
	 * sc_flagsのリセット
	 */
	sc->sc_flags = 0;

	return (0);
}
EXPORT_SYMBOL(fcudclose);


/*
 * write
 */
ssize_t
fcudwrite(struct file* p_file, const char* buf, size_t count, loff_t* f_pos)
{
	int error;
	
	PRINT_INFO("%s: fcudwrite function was invoked.\n", DRIVER_NAME);

	if (count < sizeof(fcud_cmd_t)){
		PRINT_ERR("size error %d\n", count);
		return (-EINVAL);
	}

//	disable_irq(sc->irqno);

	error = copy_from_user(&sc->sc_wcmd, buf, sizeof(fcud_cmd_t));
	if (error) {
		PRINT_ERR("copy_from_user error\n");
//		enable_irq(sc->irqno);
		return (error);
	}

	switch (sc->sc_wcmd.command) {
	case FCUD_START:
//		if (bit_is_set(sc->sc_run, sc->sc_wcmd.dmach)) {
//			PRINT_ERR("fcudwrite dma busy\n");
//			enable_irq(sc->irqno);
//			return (-EBUSY);
//		}
		PRINT_INFO("FCUD_START.\n");
		fcud_dma_start(sc);		/* DMA start */
		break;
	default:
		PRINT_ERR("%s: write command error (%d)\n", DRIVER_NAME, sc->sc_wcmd.command);
//		enable_irq(sc->irqno);
		return (-EINVAL);
		break;
	}

//	enable_irq(sc->irqno);

	return (0);
}
EXPORT_SYMBOL(fcudwrite);


/*
 * read
 */
ssize_t
fcudread(struct file* p_file, char* buf, size_t count, loff_t* f_pos)
{
	int error;
	PRINT_INFO("%s: device read\n", DRIVER_NAME);

	if (count < sizeof(fcud_cmd_t)){
		PRINT_ERR("size error %d\n", count);
		return (-EINVAL);
	}

//	disable_irq(sc->irqno);

	while (!sc->sc_rsp) {
		FLAG_SET(FSF_RSLEEP);
//		enable_irq(sc->irqno);
		error = wait_event_interruptible(sc->sc_rpl_q, !(sc->sc_flags & FSF_RSLEEP));
		if ((wait_event_interruptible(sc->sc_rpl_q, !(FLAG_CHK(FSF_RSLEEP)))) != 0){
			PRINT_DEBUG("wait error\n");
			return (-EINTR);
		}
//		disable_irq(sc->irqno);
	}

	/* read dma size */
	fcud_dma_rsp(sc);

	error = copy_to_user(buf, &sc->sc_rrsp, sizeof(fcud_cmd_t));
	if (error) {
//		enable_irq(sc->irqno);
		return (-EFAULT);
	}


//	enable_irq(sc->irqno);

	return (sizeof(fcud_cmd_t));
}
EXPORT_SYMBOL(fcudread);

/*
 * ioctl
 */
long
fcudioctl(struct file* p_flie, unsigned int cmd, unsigned long arg)
{
	u_long data;
	int error = 0;

	PRINT_INFO("%s: device ioctl\n", DRIVER_NAME);

	switch (cmd) {
	case FDIOCSCANEND:
		PRINT_INFO("ioctl FBIOSCANEND\n");
		bsw4_io(sc, FBI_OFS_IO_SCANENDCMD, 0);
		break;
	case FDIOCDMASTAT:
		PRINT_INFO("ioctl FBIOCMDSTAT\n");
		data = bsr4_io(sc, FBI_OFS_IO_DMACTL);
		error = copy_to_user((u_long *)arg, &data, sizeof(u_long));
		break;
	case FDIOCDEBUG:
		PRINT_INFO("ioctl FBIODEBUG\n");
#ifdef	FCUD_DEBUG_IOCTL
		fcud_debug_ioctl_func(sc, *(int *)data);
#endif	/* !FCUD_DEBUG_IOCTL */
		break;
	default:
		PRINT_ERR("ioctl UNKNOWN\n");
		return (-EINVAL);
	}

	return (error);
}


/*
 * intr
 */
irqreturn_t
fcudintr(void* intr)
{
	fbi_intr_t *icp = intr;
	fcud_dmah_t *dh;
	PRINT_DEBUG("------- fcudintr --------\n");

	fcud_dma_intr(sc, dh, icp->intr, FCUD_CH_RECV); /* (1) */
	fcud_dma_intr(sc, dh, icp->intr, FCUD_CH_SEND); /* (2) */
	fcud_dma_intr(sc, dh, icp->intr, FCUD_CH_SCAN); /* (3) */
	fcud_dma_intr(sc, dh, icp->intr, FCUD_CH_PLOT); /* (4) */

	/* DMA3 abort request */
	if (icp->dma3)
		fcud_dma_abort_intr(sc, dh, FCUD_CH_SEND);

	/* DMA4 abort request */
	if (icp->dma4)
		fcud_dma_abort_intr(sc, dh, FCUD_CH_RECV);

	return IRQ_HANDLED;
}
EXPORT_SYMBOL(fcudintr);

/********************************************************************/
#ifdef	FCUD_DEBUG_IOCTL
void
fcud_debug_0(fcud_t *sc)
{
	printk("\n[$Id: fcud.c,v 1.8 2006/09/11 10:28:33 maruyama Exp $]\n");

	printk("[softc]\n");
	printk("%s : %p\n", "io_vaddr", (void *)sc->io_vaddr);
	printk("%s : %p\n", "mem_vaddrr", (void *)sc->mem_vaddr);
	printk("%s : 0x%x\n", "sc_run", sc->sc_run);
	printk("%s : 0x%x\n", "sc_rsp", sc->sc_rsp);

	printk("[sc_run]\n");
	printk("%s", bit_is_set(sc->sc_run, FCUD_CH_SCAN)?"FCUD_CH_SCAN\n":"");
	printk("%s", bit_is_set(sc->sc_run, FCUD_CH_PLOT)?"FCUD_CH_PLOT\n":"");
	printk("%s", bit_is_set(sc->sc_run, FCUD_CH_SEND)?"FCUD_CH_SEND\n":"");
	printk("%s", bit_is_set(sc->sc_run, FCUD_CH_RECV)?"FCUD_CH_RECV\n":"");

	printk("[sc_flags]\n");
	printk("%s", FLAG_CHK(FSF_ISOPEN)   ? "FSF_ISOPEN\n"   : "");
	printk("%s", FLAG_CHK(FSF_RSLEEP)   ? "FSF_RSLEEP\n"   : "");
	printk("%s", FLAG_CHK(FSF_RSELWANT) ? "FSF_RSELWANT\n" : "");
}


void
fcud_debug_ioctl_func(fcud_t *sc, int arg)
{
	switch (arg) {
	default:
	case 0:
		fcud_debug_0(sc);
		break;
	}
}
#endif	/* !FCUD_DEBUG_IOCTL */

static void fcud_initial()
{	
	memset(sc, 0, sizeof(fcud_t));
	init_waitqueue_head(&sc->sc_rpl_q);
}




